package me.fullpage.tvouchers.managers;

import de.tr7zw.fullpage.nbtapi.NBTItem;
import me.fullpage.tvouchers.data.Config;
import me.fullpage.tvouchers.data.ConfigManager;
import me.fullpage.tvouchers.managers.cooldowns.CooldownManager;
import me.fullpage.tvouchers.utilities.CM;
import me.fullpage.tvouchers.utilities.TimeUtil;
import me.fullpage.tvouchers.utilities.Utils;
import org.bukkit.Bukkit;
import org.bukkit.Material;
import org.bukkit.Sound;
import org.bukkit.configuration.file.FileConfiguration;
import org.bukkit.enchantments.Enchantment;
import org.bukkit.entity.Player;
import org.bukkit.inventory.ItemFlag;
import org.bukkit.inventory.ItemStack;
import org.bukkit.inventory.meta.ItemMeta;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Set;

import static me.fullpage.tvouchers.managers.Base.string;

public class Voucher {

    private final ItemStack itemStack;

    private final String voucherName;
    private final String permission;
    private final NBTItem nbtItem;
    private final boolean gui;
    private final int cooldown;
    private final List<String> commands;
    private final String successMessage;
    private final String cooldownMessage;
    private final boolean sendTitleScreen;
    private final String money;
    private final String soundString;

    private static HashMap<String, List<String>> commandMap = new HashMap<>();
    private static HashMap<String, String> successMessageMap = new HashMap<>();
    private static HashMap<String, String> cooldownMessageMap = new HashMap<>();
    private static HashMap<String, Voucher> vouchers = new HashMap<>();


    public Voucher(ItemStack itemStack, String voucherName, String permission, boolean gui, int cooldown, List<String> commands, String successMessage, String cooldownMessage, boolean sendTitleScreen, String money, String soundString) {
        this.itemStack = itemStack;
        this.voucherName = voucherName;
        this.permission = permission != null ? permission : "";
        this.gui = gui;
        this.cooldown = cooldown;
        NBTItem nbtItem = new NBTItem(itemStack);
        nbtItem.setString("VoucherName", voucherName);
        nbtItem.setString("Permission", permission);
        nbtItem.setBoolean("GUI", gui);
        nbtItem.setInteger("Cooldown", cooldown);
        nbtItem.setString("TVouchers", Config.IDENTIFIER);
        nbtItem.setBoolean("sendTitleScreen", sendTitleScreen);
            nbtItem.setString("money", money);
        this.nbtItem = nbtItem;
        this.commands = commands;
        this.successMessage = successMessage;
        this.cooldownMessage = cooldownMessage;
        this.sendTitleScreen = sendTitleScreen;
        this.money = money;
        Sound sound;
        try {
            sound = Sound.valueOf(soundString.toUpperCase());
        } catch (Exception e) {
            sound = null;
        }
        if (sound != null) {
            nbtItem.setString("Sound", sound.name());
        }
        this.soundString = soundString;
        successMessageMap.put(voucherName.toLowerCase(), successMessage);
        cooldownMessageMap.put(voucherName.toLowerCase(), cooldownMessage);
        commandMap.put(voucherName.toLowerCase(), commands);
    }

    public Voucher(ItemStack itemStack) {
        this.itemStack = itemStack;
        this.nbtItem = new NBTItem(itemStack);
        this.voucherName = isVoucher() ? getVoucherNameFromItem() : null;
        this.gui = isVoucher() && getGUIFromItem();
        this.permission = isVoucher() ? getPermissionFromItem() : null;
        this.cooldown = isVoucher() ? getCooldownFromItem() : 0;
        this.commands = isVoucher() ? getCommandsFromItem() : new ArrayList<>();
        this.successMessage = isVoucher() ? getSuccessMessageFromItem() : Config.DEFAULT_SUCCESS_MESSAGE;
        this.cooldownMessage = isVoucher() ? getCooldownMessageFromItem() : Config.DEFAULT_COOLDOWN_MESSAGE;
        this.sendTitleScreen = isVoucher() && getTitleScreenFromItem();
        this.money = isVoucher() ? getMoneyFromVoucher() : null;
        this.soundString = isVoucher() ? getSoundFromItem() : null;

    }

    public List<String> getCommandsFromItem() {
        return commandMap.getOrDefault(this.voucherName.toLowerCase(), new ArrayList<>());
    }

    public String getMoneyFromVoucher() {
        if (!nbtItem.hasKey("money"))
            return null;
        return nbtItem.getString("money");
    }

    public String getCooldownMessageFromItem() {
        return cooldownMessageMap.getOrDefault(this.voucherName.toLowerCase(), Config.DEFAULT_COOLDOWN_MESSAGE);
    }

    public Boolean getTitleScreenFromItem() {
        if (!nbtItem.hasKey("sendTitleScreen"))
            return null;
        return nbtItem.getBoolean("sendTitleScreen");
    }

    public String getSuccessMessageFromItem() {
        return successMessageMap.getOrDefault(this.voucherName.toLowerCase(), Config.DEFAULT_SUCCESS_MESSAGE);
    }


    public boolean isVoucher() {
        return nbtItem.hasKey("TVouchers") && nbtItem.getString("TVouchers").equals(Config.IDENTIFIER);
    }

    public ItemStack getOriginalItemStack() {
        return itemStack;
    }

    public NBTItem getNbtItem() {
        return nbtItem;
    }

    public boolean isGui() {
        return gui;
    }

    public int getCooldownFromItem() {
        return nbtItem.getInteger("Cooldown");
    }

    public String getVoucherNameFromItem() {
        return nbtItem.getString("VoucherName");
    }
    public String getSoundFromItem() {
        return nbtItem.getString("Sound");
    }

    public String getPermissionFromItem() {
        return nbtItem.getString("Permission");
    }

    public boolean getGUIFromItem() {
        return nbtItem.getBoolean("GUI");
    }

    public String getPermission() {
        return Utils.isNotNullOrEmpty(permission) ? permission : getPermissionFromItem();
    }

    public String getVoucherName() {
        return Utils.isNotNullOrEmpty(voucherName) ? voucherName : getVoucherNameFromItem();
    }

    public static Set<String> getVouchersList() {
        return vouchers.keySet();
    }

    public static HashMap<String, Voucher> getVouchers() {
        return vouchers;
    }

    public static boolean voucherFromName(String voucherName) {
        return vouchers.get(voucherName.toLowerCase()) != null;
    }

    public static void updateVouchers(HashMap<String, Voucher> voucherList) {
        vouchers = voucherList;
    }

    public static void loadVouchers() {
        vouchers.clear();
        final ConfigManager voucherManager = new ConfigManager("voucher.yml");
        final FileConfiguration voucher = voucherManager.getConfig();
        for (String key : voucher.getConfigurationSection("Vouchers").getKeys(false)) {
            String path = "Vouchers." + key;

            String materialName = voucher.get(path + ".MATERIAL") != null ? voucher.getString(path + ".MATERIAL") : "STONE";
            Material material = Material.getMaterial(materialName.toUpperCase());

            int damage = voucher.get(path + ".DAMAGE") != null ? voucher.getInt(path + ".DAMAGE") : 0;
            boolean glow = voucher.get(path + ".GLOW") != null && voucher.getBoolean(path + ".GLOW");
            boolean gui = voucher.get(path + ".GUI") != null && voucher.getBoolean(path + ".GUI");
            String itemName = voucher.get(path + ".ITEM_NAME") != null ? voucher.getString(path + ".ITEM_NAME") : material.name();
            List<String> itemLore = voucher.get(path + ".ITEM_LORE") != null ? voucher.getStringList(path + ".ITEM_LORE") : new ArrayList<>();
            List<String> commands = voucher.get(path + ".COMMANDS") != null ? voucher.getStringList(path + ".COMMANDS") : new ArrayList<>();
            String permission = voucher.get(path + ".PERMISSION") != null ? voucher.getString(path + ".PERMISSION") : "";
            int cooldown = voucher.get(path + ".COOLDOWN") != null ? voucher.getInt(path + ".COOLDOWN") : 0;
            String successMessage = voucher.get(path + ".SUCCESS_MESSAGE") != null ? voucher.getString(path + ".SUCCESS_MESSAGE") : Config.DEFAULT_SUCCESS_MESSAGE;
            String cooldownMessage = voucher.get(path + ".COOLDOWN_MESSAGE") != null ? voucher.getString(path + ".COOLDOWN_MESSAGE") : Config.DEFAULT_COOLDOWN_MESSAGE;
            boolean sendTitleScreen = voucher.getBoolean(path + ".TITLE_SCREEN", false);
            String money = voucher.getString(path + ".MONEY", null);
            String soundString = voucher.getString(path + ".SOUND", null);


            ItemStack itemStack = new ItemStack(material, 1, (short) damage);
            ItemMeta itemMeta = itemStack.getItemMeta();
            itemMeta.setDisplayName(CM.colourStr(itemName));
            List<String> loreList = new ArrayList<>();
            itemLore.forEach(lore -> loreList.add(CM.colourStr(lore)));
            itemMeta.setLore(loreList);
            if (glow) {
                itemMeta.addEnchant(Enchantment.OXYGEN, 1, true);
                itemMeta.addItemFlags(ItemFlag.HIDE_PLACED_ON);
                itemMeta.addItemFlags(ItemFlag.HIDE_ENCHANTS);
                itemMeta.addItemFlags(ItemFlag.HIDE_ATTRIBUTES);
            }
            itemStack.setItemMeta(itemMeta);


            vouchers.put(key.toLowerCase(), new Voucher(itemStack, key.toLowerCase(), permission, gui, cooldown, commands, successMessage, cooldownMessage, sendTitleScreen, money, soundString));
        }
    }

    public void removeSingleItemFromHand(Player player) {
        ItemStack hand = player.getItemInHand();
        if (hand.getAmount() == 1)
            player.setItemInHand(null);
        else hand.setAmount(hand.getAmount() - 1);
    }

    public boolean checkAndPutCooldown(Player player) {
        if (getCooldownFromItem() != 0) {
            final CooldownManager cooldownManager = new CooldownManager(player, getVoucherNameFromItem());
            if (cooldownManager.isOnCooldown()) {
                player.sendMessage(string(CM.colourStr(getCooldownMessageFromItem())).replaceIgnoreCase("{TIME}", String.valueOf(Config.TIME_FORMATTED ? TimeUtil.formatTime(cooldownManager.getSecondsRemaining() * 1000) : cooldownManager.getSecondsRemaining())).get());
                return true;
            } else {
                cooldownManager.putOnCooldown(getCooldownFromItem(), System.currentTimeMillis());
                return false;
            }
        }
        return false;
    }

    public boolean isOnCooldown(Player player) {
        if (getCooldownFromItem() != 0) {
            final CooldownManager cooldownManager = new CooldownManager(player, getVoucherNameFromItem());
            if (cooldownManager.isOnCooldown()) {
                player.sendMessage(string(CM.colourStr(getCooldownMessageFromItem())).replaceIgnoreCase("{TIME}", String.valueOf(Config.TIME_FORMATTED ? TimeUtil.formatTime(cooldownManager.getSecondsRemaining() * 1000) : cooldownManager.getSecondsRemaining())).get());
                return true;
            }
        }
        return false;

    }

    public void putCooldown(Player player) {
        final CooldownManager cooldownManager = new CooldownManager(player, getVoucherNameFromItem());
        cooldownManager.putOnCooldown(getCooldownFromItem(), System.currentTimeMillis());
    }

    public void runCommands(Player player) {
        getCommandsFromItem().forEach(command -> {
            command = Utils.randomNumberFromPlaceholder(string(command).remove("/").replaceIgnoreCase("{PLAYER}", player.getName()).get());
            Bukkit.dispatchCommand(Bukkit.getConsoleSender(), command);
        });
    }

    public boolean isMoneyVoucher() {
        return isVoucher() && Utils.isNotNullOrEmpty(getMoneyFromVoucher()) && getTitleScreenFromItem() != null;
    }

}
