package me.fullpage.core.utilities;

import com.mojang.authlib.GameProfile;
import com.mojang.authlib.properties.Property;
import net.md_5.bungee.api.ChatMessageType;
import net.md_5.bungee.api.chat.TextComponent;
import org.bukkit.Bukkit;
import org.bukkit.ChatColor;
import org.bukkit.entity.Player;
import org.bukkit.inventory.ItemStack;
import org.bukkit.inventory.meta.SkullMeta;

import java.io.File;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.*;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import static me.fullpage.core.utilities.EditDistance.editDistance;
import static org.apache.commons.lang.Validate.notNull;

public class Utils {
    public static double similarity(String s1, String s2) {
        String longer = s1, shorter = s2;
        if (s1.length() < s2.length()) { // longer should always have greater length
            longer = s2;
            shorter = s1;
        }
        int longerLength = longer.length();
        if (longerLength == 0) {
            return 1.0; /* both strings are zero length */
        }
        return (longerLength - editDistance(longer, shorter)) / (double) longerLength;
    }


    public static long convertTime(long duration, TimeUnit before, TimeUnit after) {
        return after.convert(duration, before);
    }

    public static int floor(double num) {
        int floor = (int) num;
        return floor == num ? floor : floor - (int) (Double.doubleToRawLongBits(num) >>> 63);
    }

    public static void sendActionBar(Player player, String message) {
        player.spigot().sendMessage(ChatMessageType.ACTION_BAR, TextComponent.fromLegacyText(ChatColor.translateAlternateColorCodes('&', message)));
    }

    public static String formatBalance(double balance) {
        NumberFormat currency = NumberFormat.getCurrencyInstance(Locale.US);
        return currency.format(balance);
    }

    private static int getPing(Player p) {
        String v = Bukkit.getServer().getClass().getPackage().getName().replace(".", ",").split(",")[3];
        if (!p.getClass().getName().equals("org.bukkit.craftbukkit." + v + ".entity.CraftPlayer")) { //compatibility with some plugins
            p = Bukkit.getPlayer(p.getUniqueId()); //cast to org.bukkit.entity.Player
        }

        int ping;
        try {
            Object entityPlayer = p.getClass().getMethod("getHandle").invoke(p);
            ping = (int) entityPlayer.getClass().getField("ping").get(entityPlayer);
        } catch (Exception e1) {
            ping = 0;
        }

        if (ping == 0) {
            try {
                Class<?> PlayerClass = Class.forName("org.bukkit.entity.Player");
                Object Player = PlayerClass.cast(p);
                Method getPing = Player.getClass().getMethod("getPing");
                Object playerPing = getPing.invoke(Player);
                ping = (int) playerPing;
            } catch (Exception e) {
                ping = 0;
            }
        }

        if (ping == 0) {
            try {
                Class<?> CraftPlayerClass = Class.forName("org.bukkit.craftbukkit." + v + ".entity.CraftPlayer");
                Object CraftPlayer = CraftPlayerClass.cast(p);
                Method getHandle = CraftPlayer.getClass().getMethod("getHandle");
                Object EntityPlayer = getHandle.invoke(CraftPlayer);
                Field pi = EntityPlayer.getClass().getDeclaredField("ping");
                ping = pi.getInt(EntityPlayer);
            } catch (Exception e) {
                ping = 0;
            }
        }
        return ping;
    }

    public static String formatValue(double value) {
        int power;
        String suffix = " kmbt";
        String formattedNumber = "";

        NumberFormat formatter = new DecimalFormat("#,###.#");
        power = (int) StrictMath.log10(value);
        value = value / (Math.pow(10, (power / 3) * 3));
        formattedNumber = formatter.format(value);
        formattedNumber = formattedNumber + suffix.charAt(power / 3);
        return formattedNumber.length() > 4 ? formattedNumber.replaceAll("\\.[0-9]+", "") : formattedNumber;
    }

    public static ItemStack getSkull(String base64) {
        final ItemStack head = XMaterial.PLAYER_HEAD.parseItem();
        SkullMeta meta = (SkullMeta) head.getItemMeta();
        GameProfile profile = new GameProfile(UUID.randomUUID(), "");
        profile.getProperties().put("textures", new Property("textures", base64));
        Field profileField;
        try {
            profileField = meta.getClass().getDeclaredField("profile");
            profileField.setAccessible(true);
            profileField.set(meta, profile);
        } catch (IllegalArgumentException | IllegalAccessException | NoSuchFieldException | SecurityException e) {
            e.printStackTrace();
        }
        head.setItemMeta(meta);
        return head;
    }

    private static HashMap<UUID, ItemStack> headCache = new HashMap<>();

    public static ItemStack getHead(UUID id, String name) {
        final ItemStack head = XMaterial.PLAYER_HEAD.parseItem();

        final ItemStack itemStack = headCache.get(id);
        if (itemStack != null) {
            return itemStack;
        }

        SkullMeta meta = (SkullMeta) head.getItemMeta();
        try {
            meta.setOwningPlayer(Bukkit.getOfflinePlayer(id));
        } catch (NoSuchMethodError e) {
            meta.setOwner(name);
        }
        head.setItemMeta(meta);
        headCache.put(id, head);

        return head;
    }

    public static ItemStack getHead(Player player) {
        final UUID id = player.getUniqueId();
        final ItemStack head = XMaterial.PLAYER_HEAD.parseItem();
        notNull(id, "id");

        SkullMeta meta = (SkullMeta) head.getItemMeta();
        meta.setOwningPlayer(Bukkit.getOfflinePlayer(id));
        head.setItemMeta(meta);

        headCache.put(player.getUniqueId(), head);
        return head;
    }

    public static int countChar(String str, char c) {
        int count = 0;

        for (int i = 0; i < str.length(); i++) {
            if (str.charAt(i) == c)
                count++;
        }

        return count;
    }

    public static boolean isNullOrEmpty(String value) {
        return (value == null) || (value.trim().length() == 0);
    }

    public static boolean isNullOrEmpty(Object value) {

        return value == null;
    }

    public static <T> boolean isNullOrEmpty(Collection<T> collection) {

        return (collection == null) || (collection.isEmpty());
    }

    public static boolean isNullOrEmpty(Number number) {

        return (number == null) || (!(number.doubleValue() > 0));
    }

    public static boolean isNullOrEmpty(Date data) {

        return data == null;
    }

    public static <T> boolean isNullOrEmpty(Map<T, T> map) {

        return (map == null) || (map.isEmpty());
    }

    public static boolean isNullOrEmpty(File file) {

        return isNull(file) || file.length() == 0;
    }

    public static boolean isNullOrEmpty(Object[] array) {

        return (array == null) || (array.length == 0);
    }

    public static boolean isNull(Object value) {

        return value == null;
    }

    public static String convertToRoman(int input) {
        if (input < 1 || input > 3999)
            return String.valueOf(input);

        StringBuilder s = new StringBuilder();

        while (input >= 1000) {
            s.append("M");
            input -= 1000;
        }
        while (input >= 900) {
            s.append("CM");
            input -= 900;
        }
        while (input >= 500) {
            s.append("D");
            input -= 500;
        }
        while (input >= 400) {
            s.append("CD");
            input -= 400;
        }
        while (input >= 100) {
            s.append("C");
            input -= 100;
        }
        while (input >= 90) {
            s.append("XC");
            input -= 90;
        }
        while (input >= 50) {
            s.append("L");
            input -= 50;
        }
        while (input >= 40) {
            s.append("XL");
            input -= 40;
        }
        while (input >= 10) {
            s.append("X");
            input -= 10;
        }
        while (input >= 9) {
            s.append("IX");
            input -= 9;
        }
        while (input >= 5) {
            s.append("V");
            input -= 5;
        }
        while (input >= 4) {
            s.append("IV");
            input -= 4;
        }
        while (input >= 1) {
            s.append("I");
            input -= 1;
        }
        return s.toString();
    }


    public static int getBetweenAndIncluding(int lowest, int highest) {
        return ThreadLocalRandom.current().nextInt(lowest, highest + 1);
    }


    public static String getBetween(String string, String start, String end) {
        if (string.contains(start))
            string = string.substring(string.indexOf(start) + 1);
        if (string.contains(end))
            string = string.substring(0, string.indexOf(end));
        return string;
    }

    public static ItemStack glassFromNumber(int i) {
        ItemStack glass;
        switch (i) {
            case 0:
                glass = XMaterial.WHITE_STAINED_GLASS_PANE.parseItem();
            case 1:
                glass = XMaterial.ORANGE_STAINED_GLASS_PANE.parseItem();
                break;
            case 2:
                glass = XMaterial.MAGENTA_STAINED_GLASS_PANE.parseItem();
                break;
            case 3:
                glass = XMaterial.LIGHT_BLUE_STAINED_GLASS_PANE.parseItem();
                break;
            case 4:
                glass = XMaterial.YELLOW_STAINED_GLASS_PANE.parseItem();
                break;
            case 5:
                glass = XMaterial.LIME_STAINED_GLASS_PANE.parseItem();
                break;
            case 6:
                glass = XMaterial.PINK_STAINED_GLASS_PANE.parseItem();
                break;
            case 7:
                glass = XMaterial.GRAY_STAINED_GLASS_PANE.parseItem();
                break;
            case 8:
                glass = XMaterial.LIGHT_GRAY_STAINED_GLASS_PANE.parseItem();
                break;
            case 9:
                glass = XMaterial.CYAN_STAINED_GLASS_PANE.parseItem();
                break;
            case 10:
                glass = XMaterial.PURPLE_STAINED_GLASS_PANE.parseItem();
                break;
            case 11:
                glass = XMaterial.BLUE_STAINED_GLASS_PANE.parseItem();
                break;
            case 12:
                glass = XMaterial.BROWN_STAINED_GLASS_PANE.parseItem();
                break;
            case 13:
                glass = XMaterial.GREEN_STAINED_GLASS_PANE.parseItem();
                break;
            case 14:
                glass = XMaterial.RED_STAINED_GLASS_PANE.parseItem();
                break;
            case 15:
                glass = XMaterial.BLACK_STAINED_GLASS_PANE.parseItem();
                break;
            default:
                glass = XMaterial.GRAY_STAINED_GLASS_PANE.parseItem();
                break;
        }
        return glass;
    }


}
