package me.fullpage.core;

import me.fullpage.core.utilities.SString;
import org.bukkit.command.Command;
import org.bukkit.command.CommandSender;
import org.bukkit.command.ConsoleCommandSender;
import org.bukkit.command.PluginIdentifiableCommand;
import org.bukkit.entity.Player;
import org.bukkit.plugin.Plugin;
import org.jetbrains.annotations.NotNull;

import java.util.Arrays;
import java.util.UUID;

public abstract class FullpageCommand extends Command implements PluginIdentifiableCommand {


    protected CommandSender sender;
    protected Player player = null;
    protected UUID uuid = null;
    protected final String permission;
    protected Integer minimumArgs = null;
    protected Integer maximumArgs = null;
    protected boolean canConsole = true;

    public FullpageCommand(String name, String permission) {
        super(name);
        this.permission = permission;
    }

    public abstract void run(CommandSender sender, String commandLabel, String[] args);

    @Override
    public boolean execute(CommandSender sender, String commandLabel, String[] args) {


        this.sender = sender;

        if (isConsole() && !canConsole()) {
            sendOnlyPlayersMessage();
            return true;
        }

        if (!isConsole()) {
            player = (Player) sender;
            uuid = player.getUniqueId();
        }

        if ((minimumArgs != null && args.length < minimumArgs) || (maximumArgs != null && args.length > maximumArgs)) {
            sendUsageMessage();
            return true;
        }

        if (permission == null || sender.hasPermission(permission) || sender.isOp()) {
            run(sender, commandLabel, args);
        } else {
            sendNoPermissionMessage();
        }
        return true;
    }

    protected void sendNoPermissionMessage() {
        sender.sendMessage(new SString("&cYou do not have permission to use this command").colourise());
    }

    protected void sendOnlyPlayersMessage() {
        sender.sendMessage(new SString("&cOnly Players can run this command").colourise());
    }

    protected void sendMessage(String... messages) {
        Arrays.stream(messages)
                .forEach(message -> sender.sendMessage(new SString(message).colourise()));
    }

    protected void sendMessage(CommandSender sender, String... messages) {
        Arrays.stream(messages)
                .forEach(message -> sender.sendMessage(new SString(message).colourise()));
    }

    protected boolean isConsole() {
        return sender instanceof ConsoleCommandSender;
    }

    public void setCanConsole(boolean canConsole) {
        this.canConsole = canConsole;
    }

    public boolean canConsole() {
        return canConsole;
    }

    public void setMinimumArgs(Integer minimumArgs) {
        this.minimumArgs = minimumArgs;
    }

    public void setMaximumArgs(Integer maximumArgs) {
        this.maximumArgs = maximumArgs;
    }

    public void sendUsageMessage() {
        sendMessage("&cIncorrect usage, please try:");
        sendMessage("&e" + (getUsage() == null ? "&7<Cannot get usage>" : getUsage()));
    }


    public FullpageCore getCore() {
        return FullpageCore.getCore();
    }

    @NotNull
    @Override
    public Plugin getPlugin() {
        return getCore().getPlugin();
    }
}