package com.ralitski.mc.bukkit.util.commands;

import com.ralitski.mc.bukkit.util.Messenger;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import org.bukkit.command.CommandSender;

/**
 *
 * @author ralitski
 */
public class AnnotationCommandParser implements CommandExecutorEXT {
    
    public static final String HELP = "help";
    
    private Command info;
    private Messenger messenger;
    
    /**
     * The executor that is called if there are no arguments or the arguments are invalid. May be used to
     * send usage messages or etc.
     */
    private CommandExecutorEXT emptyExecutor;
    private Map<String, CommandExecutorEXT> subSections;
    
    public AnnotationCommandParser(String commandName, Messenger messenger) {
        this(new CustomCommand(commandName), messenger);
    }
    
    public AnnotationCommandParser(Command info, Messenger messenger) {
        this.info = info;
        this.messenger = messenger;
        subSections = new HashMap<>();
    }
    
    /**
     * Creates a new AnnotationCommandParser using the specified CommandExecutor
     * as its basic executor.
     *
     * @param emptyExecutor The CommandExecutor that will be called when there
     * is no sub-command specified by the user (or an invalid sub-command)
     * @param messenger The messenger that will handle prefixes when this AnnotationCommandParser sends messages
     */
    public AnnotationCommandParser(CommandExecutorEXT emptyExecutor, Messenger messenger) {
        this.info = emptyExecutor.getCommandInfo();
        this.emptyExecutor = emptyExecutor;
        this.messenger = messenger;
        subSections = new HashMap<>();
    }
    
    public void addSection(CommandExecutorEXT section) {
        subSections.put(section.getCommandInfo().name(), section);
    }
    
    @Override
    public Command getCommandInfo() {
        return info;
    }
    
    /**
     * Adds sections from the given object that have been marked to be executed by an AnnotationCommandExecutor.
     * @param o 
     */
    public void addSections(Object o) {
        Class<?> c = o.getClass();
        for(Method m : c.getDeclaredMethods()) {
            AnnotationCommandExecutor executor = AnnotationCommandExecutor.getCommandExecutor(o, m, messenger);
            if(executor != null) {
                this.addSection(executor);
            }
        }
    }

    @Override
    public boolean onCommand(CommandSender sender, org.bukkit.command.Command command, String label, String[] args) {
        //No arguments, call the empty executor
        if(args.length == 0) {
            return onEmptyCommand(sender, command, label, args);
        }
        //Find a sub-command to call
        String subCommand = args[0];
        CommandExecutorEXT subExecutor = subSections.get(subCommand);
        if(subExecutor != null) {
            //Call the sub-command with a subsection of the given args (cut off the sub-section's name)
            String[] newArgs = Arrays.copyOfRange(args, 1, args.length);
            return subExecutor.onCommand(sender, command, label, newArgs);
        } else if(HELP.equals(subCommand)) {
            //Send help message
            if(args.length > 1) {
                //h=Help for a specific sub-command
                String subCommandName = args[1];
                subExecutor = subSections.get(subCommandName);
                if(subExecutor != null) {
                    //Send the help
                    sendHelpMessage(sender, subExecutor);
                }
            } else {
                //Send the default behavior and sub-commands
                sendHelpList(sender);
            }
            return true;
        }
        
        //Call the empty executor again (no sub-section found)
        return onEmptyCommand(sender, command, label, args);
    }
    
    private void sendHelpMessage(CommandSender sender, CommandExecutorEXT command) {
        Command info = command.getCommandInfo();
        sender.sendMessage(messenger.prefix(
                "Help for \"" + this.info.name() + " " + info.name() + "\":",
                "Usage: " + info.name() + " " + info.usage()
        ));
        //Command description
        String[] description = info.description();
        if(description != null && description.length > 0)  {
            sender.sendMessage(messenger.prefix(description));
        }
        //Check sender type, and possibly permissions
        int senderType = SenderType.getSenderType(sender);
        if((info.senderTypes() & senderType) == 0) {
            //Incorrent sender type
            sender.sendMessage(messenger.prefix("You can not use this command."));
        } else {
            //They are the correct sender type, so tell them if they have permission or not
            String perm = info.permission();
            if(!perm.isEmpty() && !sender.hasPermission(perm)) {
                sender.sendMessage(messenger.prefix("You do not have permission to use this command."));
            } else {
                sender.sendMessage(messenger.prefix("You have permission to use this command."));
            }
        }
    }
    
    private void sendHelpList(CommandSender sender) {
        //Default behavior
        sender.sendMessage(messenger.prefix("Help for \"" + info.name() + "\":"));
        String[] description = info.description();
        if(description != null && description.length > 0)  {
            sender.sendMessage(messenger.prefix(description));
        }
        
        //List commands
        String[] cmdList = new String[subSections.size() + 1];
        cmdList[0] = "Sub-commands:";
        int i = 1;
        for(Entry<String, CommandExecutorEXT> commandEntry : subSections.entrySet()) {
            String name = commandEntry.getKey();
            CommandExecutorEXT subCommand = commandEntry.getValue();
            cmdList[i++] = name + " " + subCommand.getCommandInfo().usage();
        }
        sender.sendMessage(messenger.prefix(cmdList));
    }
    
    private boolean onEmptyCommand(CommandSender sender, org.bukkit.command.Command command, String label, String[] args) {
        return emptyExecutor == null ? onDefaultCommand(sender, command, label, args) : emptyExecutor.onCommand(sender, command, label, args);
    }
    
    //Reroute to automatically support certain blank behavior
    private boolean onDefaultCommand(CommandSender sender, org.bukkit.command.Command command, String label, String[] args) {
        sendHelpList(sender);
        return true;
    }

}
