package com.ralitski.mc.bukkit.util.commands;

import com.ralitski.mc.bukkit.util.Messenger;
import com.ralitski.util.Reflection;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.bukkit.ChatColor;
import org.bukkit.command.CommandSender;

/**
 *
 * @author ralitski
 */
public class AnnotationCommandExecutor implements CommandExecutorEXT {
    
    //convenience methods to silently fail when creating an AnnotationCommandExecutor
    
    public static AnnotationCommandExecutor getCommandExecutor(Object host, String commandMethod, Messenger messenger) {
        try {
            return new AnnotationCommandExecutor(host, commandMethod, messenger);
        } catch (IllegalArgumentException ex) {
            Logger.getLogger(AnnotationCommandExecutor.class.getName()).log(Level.SEVERE, null, ex);
            return null;
        }
    }
    
    public static AnnotationCommandExecutor getCommandExecutor(Object host, Method commandMethod, Messenger messenger) {
        try {
            return new AnnotationCommandExecutor(host, commandMethod, messenger);
        } catch (IllegalArgumentException ex) {
            Logger.getLogger(AnnotationCommandExecutor.class.getName()).log(Level.SEVERE, null, ex);
            return null;
        }
    }
    
    private static Class[] paramTypes = new Class[]{CommandWrapper.class};

    private Object host;
    private Method command;
    private Command commandInfo;
    private Messenger messenger;
    
    public AnnotationCommandExecutor(Object host, String commandMethod, Messenger messenger) {
        this(host, Reflection.getMethod(host.getClass(), commandMethod, paramTypes), messenger);
    }
    
    /**
     * 
     * @param host
     * @param commandMethod The method that will be invoked to call this
     * executor's commands. It must have an @Command annotation and take a
     * single parameter of type CommandWrapper. It may have any return value,
     * but if it does not return a boolean, this AnnotationCommandExecutor will
     * automatically return "true" upon successful command executions.
     * @param messenger
     */
    public AnnotationCommandExecutor(Object host, Method commandMethod, Messenger messenger) {
        this.host = host;
        this.command = commandMethod;
        this.commandInfo = commandMethod.getAnnotation(Command.class);
        if(commandInfo == null || !Arrays.equals(commandMethod.getParameterTypes(), paramTypes)) {
            throw new IllegalArgumentException("Invalid command method passed: " + commandMethod + "\nMethod must have an @Command annotation and take a single parameter of type CommandWrapper");
        }
        this.command.setAccessible(true);
        this.messenger = messenger;
    }
    
    @Override
    public Command getCommandInfo() {
        return commandInfo;
    }

    @Override
    public boolean onCommand(CommandSender sender, org.bukkit.command.Command command, String label, String[] args) {
        //Check the number of arguments
        int minArgs = commandInfo.minArgs();
        int maxArgs = commandInfo.maxArgs();
        int argLen = args.length;
        if(argLen < minArgs || (argLen > maxArgs && maxArgs >= 0)) {
            //too many arguments
            sendErrorMessage(sender, "Incorrect number of arguments. Minimum: " + minArgs + ", Maximum: " + maxArgs);
            return true;
        }
        
        //Check the SenderType
        int senderType = SenderType.getSenderType(sender);
        if((commandInfo.senderTypes() & senderType) == 0) {
            //incorrent sender type
            sendErrorMessage(sender, "You can not use this command.");
            return true;
        }
        
        String perm = commandInfo.permission();
        //Check permission
        if(!perm.isEmpty() && !sender.hasPermission(perm)) {
            //no permission
            sendErrorMessage(sender, "You do not have permission to use this command.");
            return true;
        }
        
        //Everything has been checked; execute the command
        CommandWrapper wrapper = new CommandWrapper(sender, command, label, args);
        try {
            Object o = this.command.invoke(host, wrapper);
            return o instanceof Boolean ? ((Boolean)o) : true;
        } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException ex) {
            Logger.getLogger(AnnotationCommandExecutor.class.getName()).log(Level.SEVERE, "Failed to invoke method " + this.command, ex);
            return true;
        }
    }
    
    private void sendErrorMessage(CommandSender sender, String error) {
        sender.sendMessage(messenger.prefix(ChatColor.RED + error));
    }
}
