package com.ralitski.mc.bukkit.nbt;

import com.ralitski.mc.bukkit.items.Unstable;
import net.minecraft.server.v1_8_R3.NBTTagList;

/**
 * A standardized wrapper for the nms NBTTagList class. Stores a list of tags; each tag must be of the same type.
 * 
 * @author ralitski
 */
public class TagList extends TagBase {
    
    private final NBTTagList handle;
    
    public TagList() {
        this(new NBTTagList());
    }

    public TagList(NBTTagList handle) {
        super(handle);
        this.handle = handle;
    }
    
    @Override
    public NBTTagList getHandle() {
        return handle;
    }
    
    /**
     * Returns the byte id of the tags stored in this TagList.
     * @return 
     */
    @Unstable
    public int getTagIdType() {
        //net.minecraft.server.v1_8_R3.NBTTagList
        //public int f()
        return handle.f();
    }
    
    //if you can't tell what the next few methods do, you probably shouldn't be here
    
    public void add(TagBase tag) {
        handle.add(tag.getHandle());
    }
    
    //convenience method
    public void addString(String string) {
        add(new TagString(string));
    }
    
    @Unstable
    public void set(int index, TagBase tag) {
        //net.minecraft.server.v1_8_R3.NBTTagList
        //public void a(int i, NBTBase nbtbase)
        handle.a(index, tag.getHandle());
    }
    
    public void setString(int index, String string) {
        set(index, new TagString(string));
    }
    
    @Unstable
    public TagBase remove(int index) {
        //net.minecraft.server.v1_8_R3.NBTTagList
        //public NBTBase a(int i)
        return TagBase.wrap(handle.a(index));
    }
    
    public int size() {
        return handle.size();
    }
    
    //==========================================================================
    //convenience methods from NBTTagList
    //==========================================================================
    //TODO: perhaps I will make more of these, perhaps not
    
    /**
     * Returns the TagBase stored at the given index in this TagList, or a TagEnd (not stored) if not found.
     * @param index
     * @return 
     */
    @Unstable
    public TagBase getTag(int index) {
        //net.minecraft.server.v1_8_R3.NBTTagList
        //public NBTBase g(int i)
        return TagBase.wrap(handle.g(index));
    }
    
    /**
     * Returns the TagCompound stored in this TagList at the given index, or a new (empty, not stored) one if not found.
     * @param index
     * @return 
     */
    public TagCompound getCompoundTag(int index) {
        return (TagCompound)TagBase.wrap(handle.get(index));
    }
    
    /**
     * Returns the integer array stored in this TagList at the given index, or a new (length 0, not stored) one if not found.
     * @param index
     * @return 
     */
    @Unstable
    public int[] getIntArray(int index) {
        //net.minecraft.server.v1_8_R3.NBTTagList
        //public int[] c(int i)
        return handle.c(index);
    }
    
    /**
     * Returns the double stored in this TagList at the given index, or 0 (not stored) if not found.
     * @param index
     * @return 
     */
    @Unstable
    public double getDouble(int index) {
        //net.minecraft.server.v1_8_R3.NBTTagList
        //public double d(int i)
        return handle.d(index);
    }
    
    /**
     * Returns the float stored in this TagList at the given index, or 0 (not stored) if not found.
     * @param index
     * @return 
     */
    @Unstable
    public float getFloat(int index) {
        //net.minecraft.server.v1_8_R3.NBTTagList
        //public float e(int i)
        return handle.e(index);
    }
    
    /**
     * Returns a String representing the TagBase stored in this TagList at the given index, or an empty String (not stored) if not found.
     * @param index
     * @return 
     */
    public String getString(int index) {
        return handle.getString(index);
    }

}
