package com.ralitski.mc.bukkit.nbt;

import com.ralitski.mc.bukkit.items.Unstable;
import java.util.Set;
import net.minecraft.server.v1_8_R3.NBTTagCompound;

/**
 * A standardized wrapper for the nms NBTTagCompound class. Compound tags store a map of names to tags, and provide many helpful methods for manipulation.
 * 
 * @author ralitski
 */
public class TagCompound extends TagBase {
    
    private final NBTTagCompound handle;

    public TagCompound() {
        this(new NBTTagCompound());
    }
    
    public TagCompound(NBTTagCompound base) {
        super(base);
        this.handle = base;
    }
    
    @Override
    public NBTTagCompound getHandle() {
        return handle;
    }
    
    /**
     * Returns the Set of names of the tags stored in this TagCompound
     * @return 
     */
    @Unstable
    public Set<String> getTagSet() {
        //net.minecraft.server.v1_8_R3.NBTTagCompound
        //public Set<String> c()
        return handle.c();
    }
   
   //===========================================================================
   //SET
   //===========================================================================
    
    public void set(String name, TagBase tag) {
        handle.set(name, tag.getHandle());
    }

    public void setByte(String name, byte value) {
        handle.setByte(name, value);
    }

    public void setShort(String name, short value) {
        handle.setShort(name, value);
    }

    public void setInt(String name, int value) {
        handle.setInt(name, value);
    }

    public void setLong(String name, long value) {
        handle.setLong(name, value);
    }

    public void setFloat(String name, float value) {
        handle.setFloat(name, value);
    }

    public void setDouble(String name, double value) {
        handle.setDouble(name, value);
    }

    public void setString(String name, String value) {
        handle.setString(name, value);
    }

    public void setByteArray(String name, byte[] value) {
        handle.setByteArray(name, value);
    }

    public void setIntArray(String name, int[] value) {
        handle.setIntArray(name, value);
    }

    public void setBoolean(String name, boolean value) {
        handle.setBoolean(name, value);
    }

   //===========================================================================
   //GET
   //===========================================================================

    public TagBase get(String name) {
        return TagBase.wrap(handle.get(name));
    }

    /**
     * Returns the type id of the tag stored under the given name, if it exists.
     * @param name
     * @return 
     */
    @Unstable
    public byte getTypeIdOf(String name) {
        //net.minecraft.server.v1_8_R3.NBTTagCompound
        //public byte b(String var1)
        return handle.b(name);
    }

    public boolean hasKey(String name) {
        return handle.hasKey(name);
    }

    /**
     * Determines if this TagCompound has a tag under the specified name, and if
     * so, if it is of the specified type.
     *
     * @param name
     * @param typeId
     * @return
     */
    public boolean hasKeyOfType(String name, int typeId) {
        return handle.hasKeyOfType(name, typeId);
    }

    public byte getByte(String name) {
        return handle.getByte(name);
    }

    public short getShort(String name) {
        return handle.getShort(name);
    }

    public int getInt(String name) {
        return handle.getInt(name);
    }

    public long getLong(String name) {
        return handle.getLong(name);
    }

    public float getFloat(String name) {
        return handle.getFloat(name);
    }

    public double getDouble(String name) {
        return handle.getDouble(name);
    }

    public String getString(String name) {
        return handle.getString(name);
    }

    public byte[] getByteArray(String name) {
        return handle.getByteArray(name);
    }

    public int[] getIntArray(String name) {
        return handle.getIntArray(name);
    }
   
    public TagCompound getCompound(String name) {
        NBTTagCompound compound = handle.getCompound(name);
        return compound != null ? new TagCompound(compound) : null;
    }

    /**
     * Returns the TagList stored under the specified name. If the stored TagList is not of the specified type, a new, empty, not-stored TagList is returned instead.
     * @param name
     * @param tagType
     * @return 
     */
    public TagList getList(String name, int tagType) {
        return new TagList(handle.getList(name, tagType));
    }

    public boolean getBoolean(String name) {
        return handle.getBoolean(name);
    }

    public void remove(String name) {
        handle.remove(name);
    }
   
    @Override
    public boolean isEmpty() {
        return handle.isEmpty();
    }
   
   /**
    * Performs a deep copy of the given tag's data to this tag. If there are
     * identically-named sub-compound tags, the tags in this instance are not
     * replaced; they are filled with the data from the other tag's respective
     * sub-tag (and may have their data overwritten).
     *
     * @param tag
    */
   public void deepCopy(TagCompound tag) {
       //net.minecraft.server.v1_8_R3.NBTTagCompound
       //public void a(NBTTagCompound var1)
       handle.a(tag.handle);
   }
    
}
