package com.ralitski.mc.bukkit.nbt;

import com.ralitski.mc.bukkit.items.Unstable;
import com.ralitski.util.Reflection;
import net.minecraft.server.v1_8_R3.NBTBase;
import net.minecraft.server.v1_8_R3.NBTTagByte;
import net.minecraft.server.v1_8_R3.NBTTagByteArray;
import net.minecraft.server.v1_8_R3.NBTTagCompound;
import net.minecraft.server.v1_8_R3.NBTTagDouble;
import net.minecraft.server.v1_8_R3.NBTTagEnd;
import net.minecraft.server.v1_8_R3.NBTTagFloat;
import net.minecraft.server.v1_8_R3.NBTTagInt;
import net.minecraft.server.v1_8_R3.NBTTagIntArray;
import net.minecraft.server.v1_8_R3.NBTTagLong;
import net.minecraft.server.v1_8_R3.NBTTagShort;
import net.minecraft.server.v1_8_R3.NBTTagString;

/**
 * Wrapper class for NBTBase. Superclass of all wrapped tag types.
 * 
 * @author ralitski
 */
public class TagBase {
    
    //==========================================================================
    //static data
    //==========================================================================

    /**
     * An array of names representing the different tag types, in order of byte id
     */
    //net.minecraft.server.v1_8_R3.NBTBase
    //public static final String[] a
    @Unstable
    public static final String[] TAG_NAMES = NBTBase.a;
    
    public static final byte ID_END = 0;
    public static final byte ID_BYTE = 1;
    public static final byte ID_SHORT = 2;
    public static final byte ID_INT = 3;
    public static final byte ID_LONG = 4;
    public static final byte ID_FLOAT = 5;
    public static final byte ID_DOUBLE = 6;
    public static final byte ID_BYTE_ARRAY = 7;
    public static final byte ID_STRING = 8;
    public static final byte ID_LIST = 9;
    public static final byte ID_COMPOUND = 10;
    public static final byte ID_INT_ARRAY = 11;
    
    /**
     * Wraps the given NBTBase in a TagBase based on its type.
     * @param base The NBTBase to be wrapped
     * @return A TagBase wrapping the given NBTBase, or null if none are applicable
     */
    public static TagBase wrap(NBTBase base) {
        if(base instanceof NBTTagByte) {
            //byte
            return new TagByte((NBTTagByte)base);
        } else if(base instanceof NBTTagByteArray) {
            //byte array
            return new TagByteArray((NBTTagByteArray)base);
        } else if(base instanceof NBTTagCompound) {
            //compound
            return new TagCompound((NBTTagCompound)base);
        } else if(base instanceof NBTTagDouble) {
            //double
            return new TagDouble((NBTTagDouble)base);
        } else if(base instanceof NBTTagEnd) {
            //end
            return new TagEnd((NBTTagEnd)base);
        } else if(base instanceof NBTTagFloat) {
            //float
            return new TagFloat((NBTTagFloat)base);
        } else if(base instanceof NBTTagInt) {
            //int
            return new TagInt((NBTTagInt)base);
        } else if(base instanceof NBTTagIntArray) {
            return new TagIntArray((NBTTagIntArray)base);
        } else if(base instanceof NBTTagLong) {
            //long
            return new TagLong((NBTTagLong)base);
        } else if(base instanceof NBTTagShort) {
            //short
            return new TagShort((NBTTagShort)base);
        } else if(base instanceof NBTTagString) {
            //string
            return new TagString((NBTTagString)base);
        } else return null;
    }
    
    /**
     * Creates a new NBTTag from the given id and wraps it in an appropriate TagBase.
     * @param id The id of the tag to be created
     * @return A TagBase wrapper of a type appropriate for the given id, wrapping an NBTTag created based on the given id
     */
    public static TagBase createTag(byte id) {
        NBTBase base = (NBTBase)Reflection.invokeStatic(NBTBase.class, "createTag", id);
        return wrap(base);
    }
    
    //==========================================================================
    //instance data
    //==========================================================================
    
    private final NBTBase handle;
    
    public TagBase(NBTBase handle) {
        this.handle = handle;
    }
    
    /**
     * Following bukkit naming conventions
     * @return The nms backbone of this wrapper class
     */
    public NBTBase getHandle() {
        return handle;
    }
    
    @Override
    public String toString() {
        return handle.toString();
    }
    
    /**
     * The byte id used to identify this type of tag in save files
     * @return The byte id of this tag type (implemented by the handle)
     */
    public byte getTypeId() {
        return handle.getTypeId();
    }

    @Override
    public TagBase clone() {
        return wrap(handle.clone());
    }
    
    public boolean isEmpty() {
        return false;
    }

    /**
     * Backed by the definition of equals(Object o) in NBTBase.java
     * @param object
     * @return
     */
    @Override
   public boolean equals(Object object) {
      if(!(object instanceof TagBase)) {
         return false;
      } else {
         TagBase tag = (TagBase)object;
         return handle.equals(tag.handle);
      }
   }

    @Override
   public int hashCode() {
      return this.getTypeId();
   }

   /**
    * Returns a human-readable string representing the data in this tag. If this TagBase is not an instance of TagString, this is identical to a call to {@link #toString()}.
    * @return 
    */
   @Unstable
    public String getReadableString() {
        return (String)Reflection.invokeDynamic(handle, "a_");
    }
    
    public static class TagNumber extends TagBase {
        
        private final NBTBase.NBTNumber numberHandle;
        
        public TagNumber(NBTBase.NBTNumber handle) {
            super(handle);
            this.numberHandle = handle;
        }
        
        @Override
        public NBTBase.NBTNumber getHandle() {
            return numberHandle;
        }
        
        @Unstable
        public long getLongValue() {
            return numberHandle.c();
        }
        
        @Unstable
        public int getIntValue() {
            return numberHandle.d();
        }
        
        @Unstable
        public short getShortValue() {
            return numberHandle.e();
        }
        
        @Unstable
        public byte getByteValue() {
            return numberHandle.f();
        }
        
        @Unstable
        public double getDoubleValue() {
            return numberHandle.g();
        }
        
        @Unstable
        public float getFloatValue() {
            return numberHandle.h();
        }
    }
}
