package com.ralitski.mc.bukkit.items;

import com.ralitski.mc.bukkit.nbt.TagCompound;
import java.util.HashMap;
import java.util.Map;
import org.bukkit.configuration.ConfigurationSection;
import org.bukkit.inventory.ItemStack;

/**
 * Keeps track of all of the SpecialItem instances that have been registered.
 * 
 * @author ralitski
 */
public class SpecialItemManager {
    
    //==========================================================================
    //tag ids
    //==========================================================================
    
    /**
     * The prefix on data tags that are created and used by the Special Item API
     */
    public static final String PREFIX = "$SI_";
    
    /**
     * The key used to store special item data in an ItemStack's NBTTagCompound
     */
    public static final String TAG_SPECIAL = PREFIX + "ITEM_DATA";
    
    /**
     * The key used to store the id of the SpecialItem type of a given ItemStack
     */
    public static final String TAG_ID = PREFIX + "ID";
    
    //==========================================================================
    //instance data
    //==========================================================================
    
    private final SpecialItemPlugin plugin;
    private final Map<String, SpecialItem> items;
    private final ConfigurationSection config;
    
    public SpecialItemManager(SpecialItemPlugin plugin, ConfigurationSection config) {
        this.plugin = plugin;
        items = new HashMap<>();
        this.config = config;
    }
    
    /**
     * Registers a SpecialItem type with this SpecialItemManager so that it can be found during calls to {@link #getSpecialItem(ItemStack)}.
     * @param item The SpecialItem type to register
     */
    public void registerItem(SpecialItem item) {
        items.put(item.getUniqueIdentifier(), item);
    }
    
    /**
     * Removes the registration for the given SpecialItem type. This does not remove data associated with special items that were formerly of this type.
     * @param item The type of SpecialItem to unregister
     */
    public void unregisterItem(SpecialItem item) {
        items.remove(item.getUniqueIdentifier());
    }
    
    /**
     * Gets the SpecialItem type registered under the given name, if one exists.
     * @param name The name (see {@link SpecialItem#getUniqueIdentifier()}) of the SpecialItem type
     * @return A SpecialItem that has the given name (unique identifier), if one is registered.
     */
    public SpecialItem getSpecialItem(String name) {
        return items.get(name);
    }
    
    //==========================================================================
    //SpecialItemInstance API
    //==========================================================================
    
    /**
     * Checks if the given ItemStack represents a special item, and if it does,
     * returns an instance that can be used to access the item's special data.
     *
     * @param item The ItemStack that may or may not be a special item
     * @return A SpecialItemInstance used to interact with the special item's
     * data, or null if this ItemStack does not represent a special item
     */
    public SpecialItemInstance getSpecialItem(ItemStack item) {
        TagCompound tag = getSpecialItemCompound(item);
        if(tag != null) {
            String id = tag.getString(TAG_ID);
            if(id != null) {
                SpecialItem specialItem = items.get(id);
                if(specialItem != null) {
                    return new SpecialItemInstance(item, specialItem, tag);
                }
            }
        }
        return null;
    }
    
    /**
     * Adds special item data to the given ItemStack. Note that the only data
     * this adds is the SpecialItem identifier. If the item is already special,
     * it returns the previous SpecialItem type.
     *
     * @param item The ItemStack that will become special
     * @param itemType The type of SpecialItem that the ItemStack will become
     * @return A SpecialItemInstance that can be used to interact with the
     * special item's data
     */
    public SpecialItemInstance createSpecialItem(ItemStack item, SpecialItem itemType) {
        //Check if this is already a special ItemStack
        SpecialItemInstance prevInstance = getSpecialItem(item);
        if(prevInstance != null) return prevInstance;
        
        TagCompound tag = new TagCompound();
        setSpecialItemCompound(item, tag);
        tag.setString(TAG_ID, itemType.getUniqueIdentifier());
        return new SpecialItemInstance(item, itemType, tag);
    }
    
    /**
     * Removes the special item data associated with the given ItemStack.
     * @param item The ItemStack to have its special data removed
     * @return A TagCompound containing the removed special item data
     */
    public TagCompound deleteSpecialItem(ItemStack item) {
        TagCompound base = ItemTagHelper.getTag(item);
        if(base != null) {
            TagCompound tag = base.getCompound(TAG_SPECIAL);
            base.remove(TAG_SPECIAL);
            return tag;
        }
        return null;
    }
    
    private TagCompound getSpecialItemCompound(ItemStack item) {
        TagCompound base = ItemTagHelper.getTag(item);
        //Check if the base has the TAG_SPECIAL key, otherwise it gives back an empty, not-stored compound tag
        if(base != null && base.hasKey(TAG_SPECIAL)) {
            return base.getCompound(TAG_SPECIAL);
        }
        return null;
    }
    
    private void setSpecialItemCompound(ItemStack item, TagCompound tag) {
        TagCompound base = ItemTagHelper.getTag(item);
        if(base == null) {
            base = new TagCompound();
            ItemTagHelper.setTag(item, base);
        }
        base.set(TAG_SPECIAL, tag);
    }
}
