package com.ralitski.mc.bukkit.items;

import com.ralitski.mc.bukkit.util.Messenger;
import com.ralitski.mc.bukkit.util.commands.Command;
import com.ralitski.mc.bukkit.util.commands.CommandWrapper;
import com.ralitski.mc.bukkit.util.commands.SenderType;
import java.util.ArrayList;
import java.util.List;
import org.bukkit.Bukkit;
import org.bukkit.ChatColor;
import org.bukkit.entity.Player;
import org.bukkit.inventory.ItemStack;
import org.bukkit.inventory.meta.ItemMeta;

/**
 *
 * @author ralitski
 */
public class SpecialItemCommands {

    private Messenger messenger;
    
    public SpecialItemCommands(Messenger messenger) {
        this.messenger = messenger;
    }
    
    @Command(
            name = "delete",
            description = {
                "Removes the special item data from the item in your hand.",
            },
            usage = "",
            permission = "specialitems.cmd.delete",
            senderTypes = SenderType.PLAYER
    )
    public void onDelete(CommandWrapper wrapper) {
        Player player = (Player)wrapper.getSender();
        ItemStack item = player.getItemInHand();
        SpecialItemInstance specialItem = SpecialItemInstance.getSpecialItem(item);
        if(specialItem != null) {
            //Delete le data
            specialItem.delete();
            player.sendMessage(messenger.prefix("The custom data for your held special item has been deleted."));
        } else {
            //Not even data yo
            player.sendMessage(messenger.prefix("The item you hold is not a special item."));
        }
    }
    
    @Command(
            name = "check",
            description = {
                "Checks if the item in your hand is a special item.",
            },
            usage = "",
            permission = "specialitems.cmd.check",
            senderTypes = SenderType.PLAYER
    )
    public void onCheck(CommandWrapper wrapper) {
        Player player = (Player)wrapper.getSender();
        ItemStack item = player.getItemInHand();
        SpecialItemInstance specialItem = SpecialItemInstance.getSpecialItem(item);
        if(specialItem != null) {
            //Le special
            String id = specialItem.getTag().getString(SpecialItemManager.TAG_ID);
            player.sendMessage(messenger.prefix("The item you hold is a special item.", "Its id is: " + id));
        } else {
            //Not even special
            player.sendMessage(messenger.prefix("The item you hold is not a special item."));
        }
    }
    
    @Command(
            name = "clone",
            description = {
                "Clones the item in your hand.",
            },
            usage = "",
            permission = "specialitems.cmd.clone",
            senderTypes = SenderType.PLAYER
    )
    public void onClone(CommandWrapper wrapper) {
        Player player = (Player)wrapper.getSender();
        ItemStack item = player.getItemInHand();
        player.getInventory().addItem(item.clone());
        player.sendMessage(messenger.prefix("Your item has been cloned."));
    }
    
    @Command(
            name = "rename",
            description = {
                "Renames the item in your hand.",
                "If no name is given, the item's current name is removed.",
                "You may use %<code> to specify colors and formatting.",
                "For example, %f would cause the color to be white.",
                "The color codes are: 0123456789abcdef",
                "The formatting codes are: klmnor"
            },
            usage = "[name]",
            permission = "specialitems.cmd.rename",
            senderTypes = SenderType.PLAYER
    )
    public void onRename(CommandWrapper wrapper) {
        //Get info
        Player player = (Player)wrapper.getSender();
        ItemStack item = player.getItemInHand();
        if(item != null) {
            ItemMeta meta = item.getItemMeta();
            if(meta != null) {
                //Parse name
                String[] args = wrapper.getArgs();
                String name = Messenger.combine(args);
                name = ChatColor.translateAlternateColorCodes('%', name);
                //Set
                meta.setDisplayName(name);
                item.setItemMeta(meta);
                player.sendMessage(messenger.prefix("Your item has been renamed to: " + name));
            } else {
                player.sendMessage(messenger.prefix("Could not rename item: " + item));
            }
        } else {
            player.sendMessage(messenger.prefix("You are not holding an item."));
        }
    }
    
    @Command(
            name = "lore",
            description = {
                "Adds lore to the item in your hand.",
                "If no index is given, the item's current name is removed.",
                "You may use %<code> to specify colors and formatting.",
                "For example, %f would cause the color to be white.",
                "The color codes are: 0123456789abcdef",
                "The formatting codes are: klmnor"
            },
            usage = "[index] <lore>",
            permission = "specialitems.cmd.lore",
            senderTypes = SenderType.PLAYER
    )
    public void onLore(CommandWrapper wrapper) {
        //Get info
        Player player = (Player)wrapper.getSender();
        ItemStack item = player.getItemInHand();
        if(item != null) {
            ItemMeta meta = item.getItemMeta();
            if(meta != null) {
                //Parse index and name
                String[] args = wrapper.getArgs();
                String indexString = args[0];
                if(args.length > 0) {
                    //Set lore
                    try {
                        int index = Integer.parseInt(indexString);
                        String lore = Messenger.combine(args, 1);
                        lore = ChatColor.translateAlternateColorCodes('%', lore);
                        List<String> itemLore = meta.hasLore() ? meta.getLore() : new ArrayList<>(index);
                        while(itemLore.size() <= index) {
                            itemLore.add("");
                        }
                        //Set
                        itemLore.set(index, ChatColor.RESET + lore);
                        meta.setLore(itemLore);
                        item.setItemMeta(meta);
                        //Feedback
                        player.sendMessage(messenger.prefix("Index " + index + " of your item's lore has been set to: " + lore));
                    } catch(NumberFormatException e) {
                        player.sendMessage(messenger.prefix("Invalid index " + indexString + " specified; unable to add lore."));
                    }
                } else {
                    //No args; remove lore
                    player.sendMessage(messenger.prefix("Removing your item's lore."));
                    meta.setLore(null);
                }
            } else {
                player.sendMessage(messenger.prefix("Could not modify lore of item: " + item));
            }
        } else {
            player.sendMessage(messenger.prefix("You are not holding an item."));
        }
    }
}
