 Commands
==========

Commands (CmdAPI) offers simple on-runtime command registering without plugin.yml with easy TAB auto-completion and custom placeholders.

CmdAPI is under MIT license, you can use it in commercial products, too - you have to mention this software name and it's license.



# Documentation

Documentation start there. For more details, see commented sources - CmdAPI.java, CustomCommand.java, CustomHelpTopic.java and TabPlaceHook.java.



# Built-in placeholders

| Name              | Description                          |
|:-----------------:|:-------------------------------------|
| `{players}`       | Current online players               |
| `{plugins}`       | Plugins                              |
| `{worlds}`        | Loaded worlds                        |
| `{materials}`     | Material names `Material.*`          |
| `{entitytypes}`   | Entity types `EntityType.*`          |
| `{sounds}`        | Sounds `Sound.*`                     |
| `{effects}`       | Effects `Effect.*`                   |



# Format of TAB auto-completer

Simple as:
* Different formats split with `//` characters
* More options split with / character
* For better readability, you can use multiple spaces spaces before & after // characters (but not after /)
* You can use placeholders, text, regex, or all-in-one
	* Use `.*` to match everything
	* Use `prefix_{players}_suffix` to get player list with entered prefix & sufix
	* Use `hello` to match `hello` parameter only
* You can escape `/` `{` `}` characters with `\`

For example:
* If you have got command `/hello` which accept everything, you will use:
	* `.*`
* If you have got command `/getuuid` which accept player nicknames and worlds, you will use:
	* `{players}/{worlds}`
* If you have got command `/give {player} {material} {number}`, you will use:
	* `{players} {materials} [0-9]+`
* If you have got command with 2 subcommands `/hello world {world}` and `/hello player {player}`, you will use:
	* `world {world} // player {player}`
* If you got command with with 3 subcommands `/hello help`, `/hello echo {world}` and `/hello echo {player}`, you will use:
	* `help // echo {worlds}/{players}`



# Register hook with placeholder {abc_example}

```java
		boolean hookAdded = CmdAPI.registerHook("abc", new TabPlaceholder() {

			@Override
			public Iterable<String> process(CommandSender sender, String name,
					String arg) {
				if(name.equals("example")) {
					//Do something and return collection with results
				}
				return null;
			}
			
		});
		
		if(hookAdded) {
			//Do something - hook was added
		}
```



# Register command /cmd with alias /acmd

```java
		RuntimeCommand cmd = new RuntimeCommand(
				new String[] {"cmd", "acmd"}, //Command + aliases
				"Example",                    //Description
				"/<command> <Example>",       //Usage displayed to user
				"{abc_example}"               //Usage for CmdAPI with custom placeholder
			) {

			@Override
			public boolean execute(CommandSender sender, String typed,
					String[] args) {
				//Check for arguments ...
				return true; //It's our command
			}
			
		};
		
		boolean cmdAdded = CmdAPI.registerCommand("abc", cmd);
		if(cmdAdded) {
			//Do something - command was successfuly registered
		}
```



# Basic plugin

Register placeholder `{abc_worldtypes}` and command `/cmd` with alias `/acmd`

```java
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.bukkit.ChatColor;
import org.bukkit.WorldType;
import org.bukkit.command.CommandSender;
import org.bukkit.plugin.java.JavaPlugin;

public class ABCPlugin extends JavaPlugin {

	@Override
	public void onEnable() {
		/*
		 * Register custom hook
		 * including placeholder {abc_worldtypes}
		 */
		boolean hookAdded = CmdAPI.registerHook("abc", new TabPlaceholder() {

			@Override
			public Iterable<String> process(CommandSender sender, String name,
					String arg) {
				if(name.equals("worldtypes")) {
					List<String> output = new ArrayList<String>();
					WorldType[] wtypes = WorldType.values();
					for(int i = 0, sz = wtypes.length; i < sz; i++) {
						String curType = wtypes[i].name();
						if(curType.toLowerCase().startsWith(arg)) {
							output.add(curType);
						}
					}
					Collections.sort(output, String.CASE_INSENSITIVE_ORDER); //Sort results
					return output;
				}
				return null;
			}
			
		});
		
		if(hookAdded) {
			//Do something if hook was added
			this.getLogger().info("Placeholder hook successfuly registered!");
		}
		
		
		/*
		 * Create & register custom command /cmd with alias /acmd
		 */
		RuntimeCommand cmd = new RuntimeCommand(
				new String[] {"cmd", "acmd"}, //Command + aliases
				"Simple command example",     //Description
				"/<command> <world_type>",    //Usage displayed to user
				"{abc_worldtypes}"            //Usage for CmdAPI with custom placeholder
			) {

			@Override
			public boolean execute(CommandSender sender, String typed,
					String[] args) {
				if(args.length != 1) {
					sender.sendMessage(ChatColor.RED + "Usage: " + this.getUsage() + " / " + this.getUsage(typed));
				} else {
					WorldType wtype = null;
					try {
						wtype = WorldType.getByName(args[0]);
					} catch(Exception e) {}
					if(wtype == null) {
						sender.sendMessage(ChatColor.RED + "Unknown world type: " + args[0]);
					} else {
						sender.sendMessage(ChatColor.GREEN + "Entered type " + wtype.getName() + ", ordinal: " + wtype.ordinal());
					}
				}
				return true; //It's our command
			}
			
		};
		
		boolean cmdAdded = CmdAPI.registerCommand("abc", cmd);
		if(cmdAdded) {
			//Do something if command was successfuly registered
			this.getLogger().info("Custom command successfuly registered!");
		}
	}
	
}
```
